import random
import time
import subprocess
from threading import Thread, Event
from pynput import keyboard as pynput_keyboard
from RPiKeyboardConfig import RPiKeyboardConfig

# ===== CONFIGURATION =====
IDLE_SECONDS = 5  # Seconds of inactivity before screensaver starts
NUM_BLOBS = 5     # Number of fireflies
FPS = 30          # Animation frames per second
FADE_SPEED = 5    # Firefly fade speed (higher = faster fade)

# ===== SETUP =====
kb = RPiKeyboardConfig()
all_leds = kb.get_leds()
positions = [tuple(led.matrix) for led in all_leds]
pos_to_idx = {tuple(led.matrix): led.idx for led in all_leds}

# ===== BRIGHTNESS CONTROL =====
def get_current_brightness():
    """Get current keyboard brightness"""
    try:
        result = subprocess.run(['rpi-keyboard-config', 'brightness'], 
                              capture_output=True, text=True, check=True)
        output = result.stdout.strip()
        if ':' in output:
            brightness_str = output.split(':')[1].split('(')[0].strip()
            return int(brightness_str)
        return int(output)
    except:
        return 255

def set_brightness(value):
    """Set keyboard brightness (0-255)"""
    try:
        subprocess.run(['rpi-keyboard-config', 'brightness', str(value)], 
                      check=True, capture_output=True)
    except:
        pass

def fade_brightness_out(start, steps=8):
    """Fade brightness from start to 0"""
    for i in range(steps + 1):
        brightness = int(start * (1 - i / steps))
        set_brightness(brightness)

# ===== FIREFLY ANIMATION =====
grid = {pos: 0 for pos in positions}
blobs = []

def init_blobs():
    """Initialize firefly positions and properties"""
    global grid
    blobs.clear()
    grid = {pos: 0 for pos in positions}  # Reset grid
    for i in range(NUM_BLOBS):
        r, c = random.choice(positions)
        blob = {
            "pos": [r, c],
            "vel": [random.uniform(-0.1, 0.1), random.uniform(-0.3, 0.3)],
            "hue": random.randint(0, 255),
            "age": -i * 20,  # Stagger start times for gradual fade-in
            "life": random.randint(600, 900)
        }
        blobs.append(blob)

def move_blobs():
    """Update firefly positions and handle bouncing/respawning"""
    for blob in blobs:
        blob["pos"][0] += blob["vel"][0]
        blob["pos"][1] += blob["vel"][1]
        r, c = blob["pos"]
        
        # Bounce off edges
        if r < 0 or r > max(pos[0] for pos in positions):
            blob["vel"][0] *= -1
        if c < min(pos[1] for pos in positions) or c > max(pos[1] for pos in positions):
            blob["vel"][1] *= -1
        
        blob["age"] += 1
        
        # Respawn firefly when lifetime expires
        if blob["age"] >= blob["life"]:
            r, c = random.choice(positions)
            blob.update({
                "pos": [r, c],
                "vel": [random.uniform(-0.1, 0.1), random.uniform(-0.3, 0.3)],
                "hue": random.randint(0, 255),
                "age": 0,
                "life": random.randint(600, 900)
            })

def draw_fireflies():
    """Render fireflies to keyboard LEDs"""
    # Fade all positions
    for pos in grid:
        grid[pos] = max(0, grid[pos] - FADE_SPEED)
    
    # Draw fireflies with fade-in/fade-out
    for blob in blobs:
        # Skip if not born yet (negative age for staggered fade-in)
        if blob["age"] < 0:
            continue
            
        r, c = int(round(blob["pos"][0])), int(round(blob["pos"][1]))
        if (r, c) in grid:
            if blob["age"] < 30:
                val = int(255 * blob["age"] / 30)
            elif blob["age"] > blob["life"] - 30:
                val = int(255 * (blob["life"] - blob["age"]) / 30)
            else:
                val = 255
            grid[(r, c)] = max(0, val)  # Ensure non-negative
    
    # Find visible blobs for color assignment
    visible_blobs = [b for b in blobs if b["age"] >= 0]
    
    # Update LEDs
    for pos, val in grid.items():
        idx = pos_to_idx[pos]
        if visible_blobs:
            nearest_blob = min(visible_blobs, key=lambda b: (b["pos"][0]-pos[0])**2 + (b["pos"][1]-pos[1])**2)
            hue = nearest_blob["hue"]
        else:
            hue = 0  # Default hue if no visible blobs yet
        kb.set_led_by_idx(idx=idx, colour=(hue, 255, max(0, val)))
    kb.send_leds()

# ===== SCREENSAVER CONTROL =====
last_activity = time.time()
screensaver_active = Event()
saved_preset_idx = None
saved_brightness = None

def screensaver_loop():
    """Main animation loop for firefly screensaver"""
    while True:
        if screensaver_active.is_set():
            move_blobs()
            draw_fireflies()
            time.sleep(1/FPS)
        else:
            time.sleep(0.1)

def on_key_press(key):
    """Handle key press - exit screensaver and restore preset"""
    global last_activity, saved_preset_idx, saved_brightness, kb
    last_activity = time.time()
    
    if screensaver_active.is_set():
        screensaver_active.clear()
        time.sleep(0.3)
        
        # Restore preset at full brightness instantly
        set_brightness(saved_brightness)
        time.sleep(0.2)
        kb = RPiKeyboardConfig()
        time.sleep(0.2)
        kb.set_current_preset_index(saved_preset_idx, save_index=False)
        
        saved_preset_idx = None
        saved_brightness = None

# ===== MAIN =====
if __name__ == "__main__":
    print("🔥 Firefly Screensaver Started")
    print(f"   Idle timeout: {IDLE_SECONDS} seconds")
    print(f"   Fireflies: {NUM_BLOBS}")
    print("   Press any key to exit screensaver\n")
    
    # Start keyboard listener
    listener = pynput_keyboard.Listener(on_press=on_key_press)
    listener.start()
    
    # Start screensaver animation thread
    Thread(target=screensaver_loop, daemon=True).start()
    
    # Initialize fireflies
    init_blobs()
    
    # Main idle detection loop
    try:
        while True:
            if time.time() - last_activity > IDLE_SECONDS and not screensaver_active.is_set():
                # Save current state
                saved_preset_idx = kb.get_current_preset_index()
                saved_brightness = get_current_brightness()
                
                # Fade out to black
                fade_brightness_out(saved_brightness)
                
                # Reinitialize keyboard and start screensaver
                time.sleep(0.3)
                kb = RPiKeyboardConfig()
                time.sleep(0.2)
                
                try:
                    kb.set_led_direct_effect()
                    init_blobs()  # Reset blobs each time screensaver starts
                    screensaver_active.set()
                    print("🌙 Screensaver activated")
                except Exception as e:
                    print(f"❌ Failed to start screensaver: {e}")
                    set_brightness(saved_brightness)
                    saved_preset_idx = None
                    saved_brightness = None
            
            time.sleep(0.1)
    except KeyboardInterrupt:
        print("\n👋 Screensaver stopped")
        if saved_brightness:
            set_brightness(saved_brightness)